/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "mx_auto_config.h"
#include "myriexpress.h"
#if MX_OS_WINNT
#include <windows.h>
#include "getopt.h"
#else
#include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "mx__driver_interface.h"

static int verbose;

static int 
report_board(mx_endpt_handle_t fd, int board_number, int max_endpoints, int *open_count);


void 
usage()
{
  fprintf(stderr, "Usage: mx_endpoint_info [args]\n");
  fprintf(stderr, "-b - Board number [0]\n");
  fprintf(stderr, "-h - help\n");
}


int 
main(int argc, char **argv)
{
  mx_endpt_handle_t fd;
  int board_number, found, look_for, open_count;
  uint32_t instance_count, max_instance, max_endpoints;
  mx_return_t ret;
  int c;

  verbose = 0;
  board_number = 0;
  look_for = 0;

  while ((c = getopt(argc, argv, "hvb:")) != EOF) switch(c) {
  case 'b':
    board_number = atoi(optarg);
    look_for = 1;
    break;
  case 'v':
    verbose = 1;
    break;
  case 'h':
  default:
    usage();
    exit(1);
  }

  mx_init();
  mx_set_error_handler(MX_ERRORS_RETURN);
  ret = mx_open_board(board_number, &fd);
  if (ret != MX_SUCCESS) {
    printf("open failed: %s\n", mx_strerror(ret));
    return 0;
  }

  if (mx__get_instance_count(fd, &instance_count) != 0) {
    printf("MX_GET_INSTANCE_COUNT failed\n");
  }

  if (mx__get_max_endpoints(fd, &max_endpoints) != 0) {
    printf("MX_GET_MAX_ENDPOINTS failed\n");
    exit(1);
  }

  if (!look_for)
    look_for = instance_count;

  if (mx__get_max_instance(fd, &max_instance) != 0) {
    printf("MX_GET_MAX_INSTANCE failed\n");
  }

  printf("%d Myrinet board%s installed.\n", instance_count,
	 instance_count > 1 ? "s" : "");
  printf("The MX driver is configured to support up to %d endpoints on %d boards.\n", 
	 max_endpoints, max_instance);
  open_count = 0;
  for (found = 0;
       (board_number < max_instance) && (found < look_for); 
       board_number++) {
    printf("===================================================================\n");
    found += report_board(fd, board_number, max_endpoints, &open_count);
  }

  mx__close(fd);
  printf("There are currently %d regular endpoint%s open\n", open_count, 
	 open_count == 1 ? "" : "s");
  return(0);
}

static void
null_term(char *s, size_t len)
{
  size_t i;

  
  for (i = 0; i < len; i++) {
    if (s[i] == '\0')
      return;
  }
  s[len - 1] = '\0';
}

static int 
report_board(mx_endpt_handle_t fd, int board_number, int max_endpoints, int *open_count)
{
  mx_get_opener_t x;
  int i, status;

  printf("Board #%d:\n", board_number);

  x.board_number = board_number;
  printf("Endpoint\t PID\t\t Command\t\t Info\n");
  for (i = -2; i < max_endpoints; i++) {
    char ep_name[10];
    x.endpt_number = i;
    sprintf(ep_name, i == -2 ? "<ether>" : i == -1 ? "<raw>" : "%d", i);
    status = mx__get_opener(fd, &x);
    if (status)
      return 1;
    if (x.closed) {
      if (verbose) printf("%s\t\t<closed>\n", ep_name);
      continue;
    }
    if (i == -2)
      printf("<ether>\t\tnone\t\tnone");
    else {
      printf("%s\t\t", ep_name);
      if (x.opener.pid == -1)
	printf("none\t\t");
      else if (x.opener.pid == 0)
	printf("<in-use>\t\t");
      else
	printf("%d\t\t", x.opener.pid);
      null_term(x.opener.comm, sizeof(x.opener.comm));
      printf("%s\t", x.opener.comm);
    }
    printf("\n");
    if (i >= 0)
      (*open_count)++;
  }

  return 1;
}
